/*
 * $Id: ClassMetrics.java 1.11 2005/11/05 08:33:11 dds Exp $
 *
 * (C) Copyright 2005 Diomidis Spinellis
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.oometer.smdl;

import org.apache.bcel.Constants;
import org.apache.bcel.generic.* ;

import java.util.HashSet;
import java.util.TreeSet;
import java.util.ArrayList;
import java.util.Arrays;


public class ClassMetrics
{


	/**
	 * name of the metrics class
	 */
	private String className;

	/**
	 * Weighted methods per class
	 */
	private int wmc;
	/**
	 * Number of children
	 */
	private int noc;
	/**
	 * Response for a Class
	 */
	private int rfc;
	/**
	 * Coupling between object classes
	 */
	private int cbo;
	/**
	 * Lack of cohesion in methods
	 */
	private int lcom;
	/**
	 * Number of public methods
	 */
	private int npm;
	/**
	 * The class's parent class
	 */
	private ClassMetrics parent;
	/**
	 * True if the class has been visited by the metrics gatherer
	 */
	private boolean visited;
	/**
	 * True if the class is public
	 */
	private boolean isPublicClass;
	/**
	 * Coupled classes: classes that use this class
	 */
	private HashSet<String> afferentCoupledClasses;


	
	public ClassMetrics(String className)
	{
		this.className = className;
		wmc = 0;
		noc = 0;
		cbo = 0;
		npm = 0;
		parent = null;
		visited = false;
		afferentCoupledClasses = new HashSet<String>();
	}

	/**
	 * Increment the weighted methods count
	 */
	public void incWmc()
	{
		wmc++;
	}

	/**
	 * Return the weighted methods per class metric
	 */
	public int getWmc()
	{
		return wmc;
	}

	/**
	 * Increment the number of children
	 */
	public void incNoc()
	{
		noc++;
	}

	/**
	 * Return the number of children
	 */
	public int getNoc()
	{
		return noc;
	}

	/**
	 * Increment the Response for a Class
	 */
	public void setRfc(int r)
	{
		rfc = r;
	}

	/**
	 * Return the Response for a Class
	 */
	public int getRfc()
	{
		return rfc;
	}

	/* Set the class's parent */
	public void setParent(ClassMetrics p)
	{
		parent = p;
	}

	/**
	 * Return the class's parent
	 */
	public ClassMetrics getParent()
	{
		return parent;
	}

	/**
	 * Return the depth of the class's inheritance tree
	 */
	public int computeDIT()
	{
		int i = 0;
		ClassMetrics c = parent;
		while (c != null)
		{
			c = c.getParent();
			i++;
		}
		return i;
	}

	/**
	 * Increment the coupling between object classes metric
	 */
	public void setCbo(int c)
	{
		cbo = c;
	}

	/**
	 * Return the coupling between object classes metric
	 */
	public int getCbo()
	{
		return cbo;
	}

	/**
	 * Return the class's lack of cohesion in methods metric
	 */
	public int getLcom()
	{
		return lcom;
	}

	/**
	 * Set the class's lack of cohesion in methods metric
	 */
	public void setLcom(int l)
	{
		lcom = l;
	}

	/**
	 * Return the class's afferent couplings metric
	 */
	public int getCa()
	{
		return afferentCoupledClasses.size();
	}

	/**
	 * Add a class to the set of classes that depend on this class
	 */
	public void addAfferentCoupling(String name)
	{
		afferentCoupledClasses.add(name);
	}

	/**
	 * Increment the number of public methods count
	 */
	public void incNpm()
	{
		npm++;
	}

	/**
	 * Return the number of public methods metric
	 */
	public int getNpm()
	{
		return npm;
	}

	/**
	 * Return true if the class is public
	 */
	public boolean isPublic()
	{
		return isPublicClass;
	}

	/**
	 * Call to set the class as public
	 */
	public void setPublic()
	{
		isPublicClass = true;
	}

	/**
	 * Return true if the class name is part of the Java SDK
	 */
	public static boolean isJdkClass(String s)
	{
		return (s.startsWith("java.") ||
				s.startsWith("javax.") ||
				s.startsWith("org.omg.") ||
				s.startsWith("org.w3c.dom.") ||
				s.startsWith("org.xml.sax."));
	}

	/**
	 * Return the 6 CK metrics plus Ce as a space-separated string
	 */
	public String toString()
	{
		return (
				wmc +
						" " + computeDIT() +
						" " + noc +
						" " + cbo +
						" " + rfc +
						" " + lcom +
						" " + getCa() +
						" " + npm);
	}

	public Object[] toArray()
	{
		return new Object[]{
						className,
						wmc,
						computeDIT(),
						noc,
						cbo,
						rfc,
						lcom,
						getCa(),
						npm};
	}


	public static int getNumberOfMetrics()
	{
		return 9;
	}
	public static Object[] getMetricsNames()
	{
		return new Object[]{
						"Class Name",
						"WMC",
						"DIT",
						"NOC",
						"CBO",
						"RFC",
						"LCOM",
						"CA",
						"NPM"};
	}


	public String getClassName()
	{
		return className;
	}

	public void setClassName(String className)
	{
		this.className = className;
	}

	/**
	 * Mark the instance as visited by the metrics analyzer
	 */
	public void setVisited()
	{
		visited = true;
	}

	/**
	 * Return true if the class has been visited by the metrics analyzer.
	 * Classes may appear in the collection as a result of some kind
	 * of coupling.  However, unless they are visited and analyzed,
	 * we do not want them to appear in the output results.
	 */
	public boolean isVisited()
	{
		return visited;
	}





	/* Classes encountered.
			 * Its cardinality is used for calculating the CBO.
			 */
	private HashSet<String> efferentCoupledClasses = new HashSet<String>();
	/**
	 * Methods encountered.
	 * Its cardinality is used for calculating the RFC.
	 */
	private HashSet<String> responseSet = new HashSet<String>();
	/**
	 * Use of fields in methods.
	 * Its contents are used for calculating the LCOM.
	 * We use a Tree rather than a Hash to calculate the
	 * intersection in O(n) instead of O(n*n).
	 */
	ArrayList<TreeSet<String>> methodInvokes = new ArrayList<TreeSet<String>>();


	/**
	 * Add a given class to the classes we are coupled to
	 */
	public void registerCoupling(String otherClassName, ClassMetricsContainer metricsContainer)
	{
		/* Measuring decision: don't couple to Java SDK */
		if ((MetricsCalculator.isJdkIncluded() ||
				!ClassMetrics.isJdkClass(otherClassName)) &&
				!className.equals(otherClassName))
		{
			efferentCoupledClasses.add(otherClassName);
			metricsContainer.getMetrics(otherClassName).addAfferentCoupling(className);
		}
	}

	/* Add the type's class to the classes we are coupled to */
	public void registerCoupling(Type t, ClassMetricsContainer metricsContainer)
	{
		registerCoupling(className(t), metricsContainer);
	}

	/**
	 * Return a class name associated with a type.
	 */
	static String className(Type t)
	{
		String ts = t.toString();

		if (t.getType() <= Constants.T_VOID)
		{
			return "java.PRIMITIVE";
		}
		else if (t instanceof ArrayType)
		{
			ArrayType at = (ArrayType) t;
			return className(at.getBasicType());
		}
		else
		{
			return t.toString();
		}
	}


	/* a method tries to access a field */
	void registerFieldAccess(String otherClassName, String fieldName, ClassMetricsContainer metricsContainer)
	{
		registerCoupling(otherClassName, metricsContainer);
		if (otherClassName.equals(className))
			methodInvokes.get(methodInvokes.size() - 1).add(fieldName);
	}

	/* Add a given method to our response set */
	void registerMethodInvocation(String className, String methodName, Type[] args, ClassMetricsContainer metricsContainer)
	{
		registerCoupling(className, metricsContainer);
		/* Measuring decision: calls to JDK methods are included in the RFC calculation */
		incRFC(className, methodName, args);
	}

	/**
	 * Called when encountering a method that should be included in the
	 * class's RFC.
	 */
	protected void incRFC(String otherClassName, String methodName, Type[] arguments)
	{
		String argumentList = Arrays.asList(arguments).toString();
		// remove [ ] chars from begin and end
		String args = argumentList.substring(1, argumentList.length() - 1);
		String signature = otherClassName + "." + methodName + "(" + args + ")";
		responseSet.add(signature);
	}

	/* a method tries to access a field */
	void addNewMethodInvokation()
	{
		methodInvokes.add(new TreeSet<String>());
	}

	private int computeCBO()
	{
		return efferentCoupledClasses.size();
	}

	private int computeRFC()
	{
		return responseSet.size();
	}

	private int computeLCOM()
	{
		/*
		 * Calculate LCOM  as |P| - |Q| if |P| - |Q| > 0 or 0 otherwise
		 * where
		 * P = set of all empty set intersections
		 * Q = set of all nonempty set intersections
		 */
		int lcom = 0;
		for (int i = 0; i < methodInvokes.size(); i++)
			for (int j = i + 1; j < methodInvokes.size(); j++)
			{
				/* A shallow unknown-type copy is enough */
				TreeSet<?> intersection = (TreeSet<?>) methodInvokes.get(i).clone();
				intersection.retainAll(methodInvokes.get(j));
				if (intersection.size() == 0)
					lcom++;
				else
					lcom--;
			}
		return (lcom > 0 ? lcom : 0);
	}


	public void updateMetrics()
	{
		setCbo(computeCBO());
		setRfc(computeRFC());
		setLcom(computeLCOM());
	}
}
