/*
 * $Id: ClassVisitor.java 1.19 2005/11/05 08:33:11 dds Exp $
 *
 * (C) Copyright 2005 Diomidis Spinellis
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.oometer.smdl;

import org.apache.bcel.classfile.*;
import org.apache.bcel.generic.*;
import org.apache.bcel.classfile.EmptyVisitor;

import java.lang.reflect.Modifier;

public class ClassVisitor extends EmptyVisitor
{
	/**
	 * The class being visited.
	 */
	private JavaClass visitedClass;
	/**
	 * The class's constant pool.
	 */
	private ConstantPoolGen cp;

	/**
	 * The class's fully qualified name.
	 */
	private String visitedClassName;

	/**
	 * The container where metrics for all classes are stored.
	 */
	private ClassMetricsContainer metricsContainer;

	/**
	 * The emtrics for the class being visited.
	 */
	private ClassMetrics classMetrics;

	public ClassVisitor(JavaClass jc, ClassMetricsContainer classMap)
	{
		visitedClass = jc;
		cp = new ConstantPoolGen(visitedClass.getConstantPool());
		metricsContainer = classMap;
		visitedClassName = jc.getClassName();
		classMetrics = metricsContainer.getMetrics(visitedClassName);
	}

	/**
	 * Return the class's metrics container.
	 */
	public ClassMetrics getMetrics()
	{
		return classMetrics;
	}

	public ClassMetricsContainer getMetricsContainer()
	{
		return metricsContainer;
	}

	public void start()
	{
		visitJavaClass(visitedClass);
	}


	/**
	 * Calculate the class's metrics based on its elements.
	 */
	public void visitJavaClass(JavaClass jc)
	{
		classMetrics.setVisited();
		if (jc.isPublic())
			classMetrics.setPublic();

		String super_name = jc.getSuperclassName();

		// increment the children of the parent
		ClassMetrics parentMetrics = metricsContainer.getMetrics(super_name);
		parentMetrics.incNoc();

		//avoid infinite recursion
		if (!jc.getClassName().equals("java.lang.Object"))
			classMetrics.setParent(parentMetrics);

		classMetrics.registerCoupling(super_name, metricsContainer);

		String interfaces[] = jc.getInterfaceNames();
		/* Measuring decision: couple interfaces */
		for (String inf : interfaces)
			classMetrics.registerCoupling(inf, metricsContainer);

		Field[] fields = jc.getFields();
		for (Field field : fields)
			field.accept(this);

		Method[] methods = jc.getMethods();
		for (Method method : methods)
			method.accept(this);
	}

	/**
	 * Called when a field access is encountered.
	 */
	public void visitField(Field field)
	{
		classMetrics.registerCoupling(field.getType(), metricsContainer);
	}



	public void visitMethod(Method method)
	{
		MethodGen methodGen = new MethodGen(method, visitedClass.getClassName(), cp);

		analyzeMethodSignature(methodGen);

		MethodVisitor methodVisitor = new MethodVisitor(methodGen, this);
		methodVisitor.start();
	}

	private void analyzeMethodSignature(MethodGen methodGen)
	{
		Method method = methodGen.getMethod();

		Type result_type = methodGen.getReturnType();
		classMetrics.registerCoupling(result_type, metricsContainer);

		Type[] argTypes = methodGen.getArgumentTypes();
		for (Type argType : argTypes)
			classMetrics.registerCoupling(argType, metricsContainer);

		String[] exceptions = methodGen.getExceptions();
		for (String exception : exceptions)
			classMetrics.registerCoupling(exception, metricsContainer);

		/* Measuring decision: A class's own methods contribute to its RFC */
		classMetrics.incRFC(visitedClassName, method.getName(), argTypes);

		classMetrics.incWmc();

		if (Modifier.isPublic(method.getModifiers()))
			classMetrics.incNpm();

		classMetrics.addNewMethodInvokation();
	}


	public void end()
	{
		classMetrics.updateMetrics();
	}
}
