/*
 * $Id: MethodVisitor.java 1.8 2005/10/09 15:36:08 dds Exp $
 *
 * (C) Copyright 2005 Diomidis Spinellis
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.oometer.smdl;

import org.apache.bcel.Constants;
import org.apache.bcel.generic.*;


class MethodVisitor extends EmptyVisitor {
    /**
     * Method generation template.
     */
    private MethodGen methodGen;
    /* The class's constant pool. */
    private ConstantPoolGen constantPoolGen;
    /**
     * The visitor of the class the method visitor is in.
     */
    private ClassVisitor classVisitor;
    private ClassMetrics classMetrics;
    private ClassMetricsContainer metricsContainer;


    /**
     * Constructor.
     */
    public MethodVisitor(MethodGen m, ClassVisitor c) {
        methodGen = m;
        classVisitor = c;
        constantPoolGen = methodGen.getConstantPool();
        classMetrics = c.getMetrics();
        metricsContainer = c.getMetricsContainer();
    }

    /**
     * Start the method's visit.
     */
    public void start() {
        if (!methodGen.isAbstract() && !methodGen.isNative()) {
            for (InstructionHandle ih = methodGen.getInstructionList().getStart();
                 ih != null; ih = ih.getNext()) {
                Instruction i = ih.getInstruction();

                if (!visitInstruction(i)) {
                    i.accept(this);
                }
            }
            updateExceptionHandlers();
        }
    }

    /**
     * Visit a single instruction.
     *
     * @param i
     * @return true if the instruction can be visited
     */
    private boolean visitInstruction(Instruction i) {
        short opcode = i.getOpcode();

        return ((InstructionConstants.INSTRUCTIONS[opcode] != null) &&
                !(i instanceof ConstantPushInstruction) &&
                !(i instanceof ReturnInstruction));
    }

    /**
     * Local variable use.
     */
    public void visitLocalVariableInstruction(LocalVariableInstruction i) {
        if (i.getOpcode() != Constants.IINC) {
            metricsContainer = classVisitor.getMetricsContainer();
        }

        classMetrics.registerCoupling(
                i.getType(constantPoolGen),
                metricsContainer);
    }

    /**
     * Array use.
     */
    public void visitArrayInstruction(ArrayInstruction i) {
        classMetrics.registerCoupling(
                i.getType(constantPoolGen),
                metricsContainer);
    }

    /**
     * Field access.
     */
    public void visitFieldInstruction(FieldInstruction i) {
        classMetrics.registerFieldAccess(
                i.getReferenceType(constantPoolGen).getClass().getName(),
                i.getFieldName(constantPoolGen),
                metricsContainer);

        classMetrics.registerCoupling(
                i.getFieldType(constantPoolGen),
                metricsContainer);
    }

    /**
     * Method invocation.
     */
    public void visitInvokeInstruction(InvokeInstruction i) {
        Type[] argTypes = i.getArgumentTypes(constantPoolGen);
        for (Type argType : argTypes) {
            classMetrics.registerCoupling(
                    argType,
                    metricsContainer);
        }

        classMetrics.registerCoupling(
                i.getReturnType(constantPoolGen),
                metricsContainer);

        /* Measuring decision: measure overloaded methods separately */
        classMetrics.registerMethodInvocation(
                i.getReferenceType(constantPoolGen).getClass().getName(),
                i.getMethodName(constantPoolGen),
                argTypes,
                metricsContainer);
    }

    /**
     * Visit an instanceof instruction.
     */
    public void visitINSTANCEOF(INSTANCEOF i) {
        classMetrics.registerCoupling(i.getType(constantPoolGen), metricsContainer);
    }

    /**
     * Visit checklast instruction.
     */
    public void visitCHECKCAST(CHECKCAST i) {
        classMetrics.registerCoupling(i.getType(constantPoolGen), metricsContainer);
    }

    /**
     * Visit return instruction.
     */
    public void visitReturnInstruction(ReturnInstruction i) {
        classMetrics.registerCoupling(i.getType(constantPoolGen), metricsContainer);
    }

    /**
     * Visit the method's exception handlers.
     */
    private void updateExceptionHandlers() {
        CodeExceptionGen[] handlers = methodGen.getExceptionHandlers();

        /* Measuring decision: couple exceptions */
        for (CodeExceptionGen handler : handlers) {
            Type t = handler.getCatchType();
            if (t != null) {
                classMetrics.registerCoupling(t, metricsContainer);
            }
		}
	}
}
